#!/usr/bin/perl

package SQL::fetchable;
use strict;
use Data::Dumper;
use Encode;

###############################################################################
sub new {
	my ($class,$aref)=@_;
	my $self={
		items=>$aref || [],
		itemindex=>0
	};
	bless($self,$class);
	return $self;
}

sub push {
	my ($self,@data)=@_;
#	foreach (@data) {
#		SQL::uc_href($_);
#	}
	push(@{$self->{items}},@data);
}

sub fetchrow_hashref {
	my ($self)=@_;
	return undef unless @{$self->{items}}>$self->{itemindex};
	$self->{itemindex}++;
	return $self->{items}->[$self->{itemindex}-1];
}

package SQL;
use strict;
use Data::Dumper;
use Storable qw/freeze thaw/;
my $cache={};

###############################################################################
sub new {
	my $class=shift;
	my ($database,$hostname,$port,$driver,$DBlogin,$DBpassword,$commit,$use_uc)=@_;
	$commit=1 unless defined $commit;
	my $self={};
	my $dsn;
	if ($driver eq "Pg") {
		$dsn="DBI:$driver:dbname=$database";$dsn.=";host=$hostname;port=$port" if defined $hostname && defined $port;
	} elsif ($driver eq "Oracle") {
		$dsn="dbi:$driver:$database";
	} elsif ($driver eq "SQLite") {
		$dsn="dbi:$driver:dbname=$database";
	} else {
		$dsn="DBI:$driver:database=$database";$dsn.=";host=$hostname;port=$port" if defined $hostname && defined $port;
	}
	bless($self,$class);
	$self->{dbh}=DBI->connect($dsn,$DBlogin,$DBpassword,{AutoCommit=>$commit});
	die "Could not connect to database ($dsn,$DBlogin,$DBpassword)\n" unless $self->{dbh};
#	$self->{dbh}->{RaiseError}=1;
	$self->{sql}={};
	$self->{sqldate}={};
	$self->{db_current}=$DBlogin;
	$self->{driver}=$driver;
	$self->{dynamic}={};
	$self->{prefix}="";
#	$self->{dbh}->{LongTruncOk}=1;
#	$self->{dbh}->{LongReadLen}=65536;
	$self->{use_uc}=$use_uc||0;
	return $self;
}

###############################################################################
sub find_file {
	my ($self,$sql)=@_;
	return $self->{prefix}.$sql.".sql";
}

###############################################################################
sub last_insert_id {
	my ($self,$tablename)=@_;
	return $self->{dbh}->last_insert_id(undef,undef,$tablename,undef);
}

###############################################################################
sub load_sql {
	my ($self,$sql,$pparams)=@_;
	my $text;
	my $fname=$self->find_file($sql);
	if ($self->{sqldate}->{$sql}) {
		my $time=(stat($fname))[9];
		return if $time==$self->{sqldate}->{$sql};
	}
	if (-f $fname) {
		my $time=(stat($fname))[9];
		$self->{sqldate}->{$sql}=$time;
	}
	
	open IN,"$fname" or die "No $fname file found !\n";
	{local $/;$text=<IN>;}
	close IN;
	if ($self->{driver} ne "Oracle") {
		$text=~s/\s+$//;
		$text.=";" unless $text=~/\;$/;
		$text.="\n";
	}
	$self->{sql}->{$sql}=$self->{dbh}->prepare($text,$pparams);
	die $self->{dbh}->errstr if $self->{dbh}->errstr;
	if ($self->{sql}->{$sql}) {
#		$self->{sql}->{$sql}->{LongTruncOk}=1;
#		$self->{sql}->{$sql}->{LongReadLen}=65536;
	} else {
		die "SQL::load_sql($sql) error, file: $fname\n";
	}
}

###############################################################################
sub pseudo_load_sql {
	my ($self,$sql,$text)=@_;
	$cache->{$sql}=$text unless defined $cache->{$sql};
	$self->{sql}->{$sql}=$self->{dbh}->prepare($text);
	unless ($self->{sql}->{$sql}) {
		die "File: $sql\n";
	}
}

###############################################################################
sub execute {
	my ($self,$sql,@vars)=@_;
#	warn join(',',@vars)."\n";
	unless (defined $self->{sql}->{$sql}) {
		$self->load_sql($sql);
	}
	if (defined $self->{sql}->{$sql}) {
		$self->{sql}->{$sql}->execute(@vars) or die "Could not execute $sql (".join(",",@vars).") - ".($self->{dbh}->errstr())."\n";
	} else {
		die "Not found in hashes: $sql\n";
	}
	return $self->{sql}->{$sql};
}
###############################################################################
sub execute_bind {
	my ($self,$sql,$href,$href_inout,$types,$no_errors,$pparams)=@_;
	$self->load_sql($sql,$pparams);# unless defined $self->{sql}->{$sql} || defined $self->{dynamic}->{$sql};
	my $result;
	unless (defined $self->{sql}->{$sql}) {
		die "No $sql";
	}
	if ($href_inout) {
		foreach my $k (keys %$href_inout) {
			my $l=length($href_inout->{$k});
			$l=20 if $l<20;
			
			$self->{sql}->{$sql}->bind_param_inout($k,\ $href_inout->{$k},$l,$types->{$k}||undef);
		}
	}
	foreach my $k (keys %$href) {
		if ($k=~/^:inout_/) {
			my $l=length($href->{$k});
			$l=20 if $l<20;
			$self->{sql}->{$sql}->bind_param_inout($k,\ $href->{$k},$l,$types->{$k}||undef);
		} else {
			$self->{sql}->{$sql}->bind_param($k,$href->{$k},$types->{$k}||undef);
		}
	}
	$result=$self->{sql}->{$sql}->execute();
	unless ($result || $no_errors) {
		my $r="";
		while (my ($k,$v)=each %$href) {
			my $v2=$v;
			$v2="[undef]" unless defined $v2;
			Encode::_utf8_off($v2);
			$r.="," if $r;$r.="$k=>$v2";
		}
		while (my ($k,$v)=each %$href_inout) {
			my $v2=$v;
			$v2="[undef]" unless defined $v2;
			Encode::_utf8_off($v2);
			$r.="," if $r;$r.="$k=>$v2";
		}
		my $r2=$self->{dbh}->errstr();
		Encode::_utf8_off($r2);
		die "Could not execute $sql {$r}:\n\nОшибка:\n$r2\n";
	}
	return ($self->{sql}->{$sql},$result);
}

###############################################################################
sub execute_and_fetch {
	my ($self,$sql,@vars)=@_;
	my $sth=$self->execute($sql,@vars);
	my @arr;
	my $t=1;
	while (defined $t) {
		$t=$sth->fetchrow_hashref();
		if (defined $t) {
			push(@arr,thaw freeze $t);
		}
	}
	$sth=undef;
	return @arr;
}

###############################################################################
sub execute_bind_from_query {
	my ($self,$sql,$href,$query,$href_inout)=@_;
	my $pparams=undef;
	$self->load_sql($sql,$pparams);
	my $result;
	die unless defined $self->{sql}->{$sql};
	my @params;
	foreach my $p (keys %{$self->{sql}->{$sql}->{ParamValues}}) {
		my $p2=$p;
		$p2=~s/://;
		if ($href_inout->{$p2}) {
			my $l=length($href_inout->{$p2});
			$l=20 if $l<20;
			$self->{sql}->{$sql}->bind_param_inout($p,$href_inout->{$p2},$l);
		} elsif ($href->{$p2}) {
			$self->{sql}->{$sql}->bind_param($p,$href->{$p2});
		} else {
			my $a=$query->param($p2);
			$self->{sql}->{$sql}->bind_param($p,$a);
		}
	}
	$result=$self->{sql}->{$sql}->execute();
	unless ($result) {
		my $r="";
		while (my ($k,$v)=each %$href) {
			my $v2=$v;
			$v2="[undef]" unless defined $v2;
			Encode::_utf8_off($v2);
			$r.="," if $r;$r.="$k=>$v2";
		}
		while (my ($k,$v)=each %$href_inout) {
			my $v2=$v;
			$v2="[undef]" unless defined $v2;
			Encode::_utf8_off($v2);
			$r.="," if $r;$r.="$k=>$v2";
		}
		my $r2=$self->{dbh}->errstr();
			Encode::_utf8_off($r2);
		die "Could not execute $sql {$r}:\n\nОшибка:\n$r2\n";
	}
	return ($self->{sql}->{$sql},$result);

}

###############################################################################
sub execute_bind_and_fetch_single {
	my ($self,$sqltext,$href,$query)=@_;
	my ($sth,$err)=$self->execute_bind_single($sqltext,$href,$query);
	my @arr;
	my $t=1;
	while (defined $t) {
		$t=$sth->fetchrow_hashref();
		if (defined $t) {
			push(@arr,thaw freeze $t);
		}
	}
	return @arr;
}

###############################################################################
sub execute_bind_single {
	my ($self,$sqltext,$href,$query,$href_inout)=@_;
	my $pparams=undef;
	my $sql=$self->{dbh}->prepare($sqltext);
	my $result;
	if ($sql) {
		my @params;
		foreach my $p (keys %{$sql->{ParamValues}}) {
			my $p2=$p;
			$p2=~s/://;

			if ($href_inout->{$p2}) {
				my $l=length($href_inout->{$p2});
				$l=20 if $l<20;
				$sql->bind_param_inout($p,\ $href_inout->{$p2},$l,$href_inout->{$p2."_type"}||undef);
			} elsif ($href->{$p2}) {
				$sql->bind_param($p,$href->{$p2},$href->{$p2."_type"}||undef);
			} else {
				my $a=$query->param($p2);
				$sql->bind_param($p,$a,$href->{$p2."_type"}||undef);
			}
		}
		$result=$sql->execute() if $sql;
	}
	unless ($result) {
		my $r="";
		while (my ($k,$v)=each %$href) {
			my $v2=$v;
			$v2="[undef]" unless defined $v2;
			Encode::_utf8_off($v2);
			$r.="," if $r;$r.="$k=>$v2";
		}
		my $r2=$self->{dbh}->errstr();
		Encode::_utf8_off($r2);
		die "Could not execute $sqltext {$r}:\n\nОшибка:\n$r2\n";
	}
	return ($sql,$result);

}

###############################################################################
sub execute_bind_and_fetch {
	my ($self,$sql,$href)=@_;
	my ($sth,$result)=$self->execute_bind($sql,$href);
	my @arr;
	my $t=1;
	while (defined $t) {
		$t=$sth->fetchrow_hashref();
		if (defined $t) {
			push(@arr,thaw freeze $t);
		}
	}
	$sth=undef;
	return @arr;
}

###############################################################################
sub execute_bind_and_fetch_one {
	my ($self,$sql,$href)=@_;
	my ($sth,$result)=$self->execute_bind($sql,$href);
	my $r=$sth->fetchrow_hashref();
	$sth->finish();
	return $r;
}

###############################################################################
sub execute_and_fetch_one {
	my ($self,$sql,@vars)=@_;
	my $sth=$self->execute($sql,@vars);
	my $r=$sth->fetchrow_hashref();
	$sth->finish();
	return $r;
}

###############################################################################
sub DESTROY {
	my ($self)=@_;
	foreach (keys %{$self->{sql}}) {
		$self->{sql}->{$_}=undef;
	}
	$self->{dbh}->disconnect() if $self->{dbh};
}

###############################################################################
sub db_switch {
	my ($self,$newdb)=@_;
	my $sql="::"."db_switch_".$newdb."::";
	my $text;
	if ($self->{driver} eq "Oracle") {
		$text="ALTER SESSION SET CURRENT_SCHEMA=$newdb";
		$self->pseudo_load_sql($sql,$text) unless $self->{sql}->{$sql};
		$self->execute($sql);
	} else {
		die "SQL::db_switch() - Can not switch!\n";
	}
	$self->{db_current}=$newdb;
}

###############################################################################
sub db_current {
	my ($self)=@_;
	return $self->{db_current};
}

###############################################################################
sub output_one_xml {
	my ($self,$l,$row)=@_;
	my $r="";
	$r.="<$row>\n" if $row;
	while (my ($k,$v)=each %$l) {
		next unless defined $v;
		$k=uc($k);
		$r.="<$k>";
		$v=~s/&/&amp;/g;
		$v=~s/</&lt;/g;
		$v=~s/>/&gt;/g;
		$r.=$v;
		$r.="</$k>\n";
	}
	$r.="</$row>\n" if $row;
	return $r;
}

###############################################################################
sub output_xml {
	my ($self,$name,$row,@params)=@_;
	my $sth=$self->execute($name,@params);
	my $r="";
	my $l;
	while ($l=$sth->fetchrow_hashref()) {
		$r.="<$row>\n" if $row;
		while (my ($k,$v)=each %$l) {
			$v="" unless defined $v;
			$k=lc($k);
			$r.="<$k>";
			$v=~s/&/&amp;/g;
			$v=~s/</&lt;/g;
			$v=~s/>/&gt;/g;
			$r.=$v;
			$r.="</$k>\n";
		}
		$r.="</$row>\n" if $row;
	}
	return $r;
}

###############################################################################

"Foreach ne furichit....";

