#include <stdio.h>
#include <atoms/text8.h>
#include <atoms/text32.h>
#include <atoms/blob.h>
#include <atoms/array.h>
#include <atoms/funcs.h>
#include <atoms/utf.h>

uint8 zero_ptr_uint8[1]={0};

/*****************************************************************************/
Text8::Text8()
{
	size_bytes=0;
	length=0;
	text=zero_ptr_uint8;
}

/*****************************************************************************/
Text8::Text8(Text*txt): Text()
{
	length=0;
	if (dynamic_cast<Text8*>(txt)) {
		init_from_text8(dynamic_cast<Text8*>(txt));
	} else if (dynamic_cast<Text32*>(txt)) {
		init_from_text32(dynamic_cast<Text32*>(txt));
	} else if (dynamic_cast<Blob*>(txt)) {
		init_from_blob(dynamic_cast<Blob*>(txt));
	}
}

/*****************************************************************************/
Text8::Text8(const char *chr)
{
	size_bytes=strlen(chr);
	if (size_bytes) {
		text=(uint8*)malloc(sizeof(uint8)*size_bytes+1);
		memmove(text,chr,sizeof(uint8)*size_bytes);
		text[size_bytes]=0;
		length=calc_length(text,size_bytes);
	} else {
		text=zero_ptr_uint8;
		length=0;
	}
}

/*****************************************************************************/
Text8::Text8(const char *chr,uint32 _size_bytes)
{
	size_bytes=_size_bytes;
	if (size_bytes) {
//		printf("Text8: %d %d\n", sizeof(uint8), size_bytes);
		text=(uint8*)malloc(sizeof(uint8)*size_bytes+1);
		memmove(text,chr,sizeof(uint8)*size_bytes);
		text[size_bytes]=0;
		length=calc_length(text,size_bytes);
	} else {
		text=zero_ptr_uint8;
		length=0;
	}
//	printf ("Text8+: %d\n" , length);
}

/*****************************************************************************/
Text8::~Text8()
{
	if (length) free(text);
//	if (text) free(text);
}

/*****************************************************************************/
void Text8::init_from_text8(Text8 *txt)
{
	size_bytes=txt->size_bytes;
	length=txt->length;
	if (size_bytes) {
		text=(uint8*)malloc(sizeof(uint8)*size_bytes+1);
		memmove(text,txt->text,sizeof(uint8)*size_bytes);
		text[size_bytes]=0;
	}
}

/*****************************************************************************/
void Text8::init_from_text32(Text32 *txt)
{
	length=txt->length;
	if (length) {
		size_bytes=calc_size_bytes(txt->text,txt->length);
		text=(uint8*)malloc(sizeof(uint8)*size_bytes+1);
		copy_utf32_to_utf8(txt->text,text,txt->length);
		text[size_bytes]=0;
	} else {
		text=zero_ptr_uint8;
		size_bytes=0;
	}
}

/*****************************************************************************/
void Text8::init_from_blob(Blob *txt)
{
	assert(0);
}

/*****************************************************************************/
uint32 Text8::calc_bytes_by_pos(uint32 pos)
{
	if (pos>=length) return size_bytes;
	uint32 i=0;
	while (pos) {
		if (text[i]<0x10000000) {
			i++;pos--;
		} else if (text[i]<0x11100000) {
			i+=2;pos--;
		} else if (text[i]<0x11110000) {
			i+=3;pos--;
		} else if (text[i]<0x11111000) {
			i+=4;pos--;
		} else if (text[i]<0x11111100) {
			i+=5;pos--;
		} else if (text[i]<0x11111110) {
			i+=6;pos--;
		} else {
			i++;
		}
	}
	return i;
}

/*****************************************************************************/
Text *Text8::concat(Text *txt1)
{
	Text8 *r=new Text8(this);
	r->add(txt1);
	return r;
}

/*****************************************************************************/
void Text8::add(const char *txt1)
{
	if (txt1==NULL) return;
/*	Text8 *t=new Text8(txt1);
	t->add_ref();
	add(t);
	t->del_ref();*/
	add_bytes((const uint8*)txt1,strlen(txt1));
}

/*****************************************************************************/
void Text8::add_bytes(const uint8* buf, uint32 len)
{
	uint8 *tmp=text;
	text=(uint8*)malloc(sizeof(uint8)*(size_bytes+len)+1);
	memmove(text,tmp,sizeof(uint8)*size_bytes);
	if (length) free(tmp);

	memmove(text+sizeof(uint8)*size_bytes,buf,sizeof(uint8)*len);
	size_bytes+=len;
	length=calc_length(text,size_bytes);
	text[size_bytes]=0;
}

/*****************************************************************************/
void Text8::add(Text *txt1)
{
	if (txt1==NULL) return;
	Text8 *txt=text_to_text8(txt1);
	txt->add_ref();
	uint8 *tmp=text;
	text=(uint8*)malloc(sizeof(uint8)*(size_bytes+txt->size_bytes)+1);
	memmove(text,tmp,sizeof(uint8)*size_bytes);
	if (length) free(tmp);

	memmove(text+sizeof(uint8)*size_bytes,txt->text,sizeof(uint8)*txt->size_bytes);
	size_bytes+=txt->size_bytes;
	length+=txt->length;
	text[size_bytes]=0;

	txt->del_ref();
}

/*****************************************************************************/
Text *Text8::substr(uint32 start,uint32 len)
{
	if (start>=length) {
		return new Text8();
	}
	if (start+len>=length) {
		len=length-start;
	}
	uint32 p1=calc_bytes_by_pos(start);
	uint32 p2=calc_bytes_by_pos(start+len);

	// TODO: char!=uint8	
	return new Text8((char*)(text+p1),p2-p1);
}

/*****************************************************************************/
const char *Text8::classname()
{
	return "Text8";
}

/*****************************************************************************/
int8 Text8::cmp(Object *txt1)
{
	if (txt1==this) return 0;
	Text8 *txt=dynamic_cast<Text8*>(txt1);
	bool del=false;
	if (txt==NULL) {
		Text *tmp=dynamic_cast<Text*>(txt1);
		if (!tmp) return Object::cmp(txt1);
		txt=new Text8(tmp);
		txt->add_ref();
		del=true;
	}
	int8 rv=memcmp2(text,size_bytes,txt->text,txt->size_bytes);
	if (del) txt->del_ref();
	return rv;
}

/*****************************************************************************/
int8 Text8::cmp(uint8 *a1,uint8 *a2,uint32 size1,uint32 size2)
{
	uint32 s=(size1<size2)?size1:size2;
	int8 r=memcmp(a1,a2,s);
	int8 rv=0;
	if (r>0) {
		return 1;
	} else if (r<0) {
		return -1;
	} else {
		if (size1<size2) return -1;
		if (size1>size2) return 1;
	}
	return 0;
}


/*****************************************************************************/
char *Text8::get_chars()
{
	char *s=(char*)malloc(sizeof(char)*(size_bytes+1));
	// TODO if uint8!=char
	// TODO: national
	memmove(s,text,sizeof(uint8)*size_bytes);
	s[size_bytes]=0;
	return s;
}

/*****************************************************************************/
uint32 Text8::get_char(uint32 pos)
{
	if (pos>=length) return (uint32)-1;
	uint32 p=calc_bytes_by_pos(pos);
	return symbol(text+p, length-p);
}

/*****************************************************************************/
int32 Text8::index_of(Text *txt1,uint32 pos)
{
	Text8 *txt=text_to_text8(txt1);
	int32 rv=-1;
	txt->add_ref();
	uint32 offset=calc_bytes_by_pos(pos);
	while (size_bytes-offset>=txt->size_bytes && rv==-1) {
		if (memcmp(text+offset,txt->text,txt->size_bytes)==0) {
			rv=pos;
		} else {
			pos++;
			if ((text[offset]&0x10000000)==0) {
				offset++;
			} else if ((text[offset]&0x11100000)==0x11000000) {
				offset+=2;
			} else if ((text[offset]&0x11110000)==0x11100000) {
				offset+=3;
			} else if ((text[offset]&0x11111000)==0x11110000) {
				offset+=4;
			} else if ((text[offset]&0x11111100)==0x11111000) {
				offset+=5;
			} else if ((text[offset]&0x11111110)==0x11111100) {
				offset+=6;
			}
		}
	}
	txt->del_ref();
	return rv;
}

/*****************************************************************************/
Array *Text8::split(Text *substr)
{
	Text8 *txt=text_to_text8(substr);
	txt->add_ref();
	uint32 prevoffset=0;
	uint32 offset=0;
	Array *a=new Array();
	while (size_bytes-offset>=txt->size_bytes) {
		if (memcmp(text+offset,txt->text,txt->size_bytes)==0) {
			a->push(new Text8((char*)(text+prevoffset),offset-prevoffset));
			offset+=txt->size_bytes;
			prevoffset=offset;
		} else {
			if ((text[offset]&0x10000000)==0) {
				offset++;
			} else if ((text[offset]&0x11100000)==0x11000000) {
				offset+=2;
			} else if ((text[offset]&0x11110000)==0x11100000) {
				offset+=3;
			} else if ((text[offset]&0x11111000)==0x11110000) {
				offset+=4;
			} else if ((text[offset]&0x11111100)==0x11111000) {
				offset+=5;
			} else if ((text[offset]&0x11111110)==0x11111100) {
				offset+=6;
			}
		}
	}
	a->push(new Text8((char*)(text+prevoffset),size_bytes-prevoffset));
	txt->del_ref();
	return a;
}

/*****************************************************************************/
bool Text8::starts_with(Text *txt1)
{
	Text8 *txt=text_to_text8(txt1);
	txt->add_ref();
	bool flag=false;
	if (txt->size_bytes<=size_bytes) {
		flag=memcmp(text,txt->text,txt->size_bytes)?false:true;
	}
	txt->del_ref();
	return flag;
}

/*****************************************************************************/
char *Text8::ptr()
{
	return (char*)text;
}

/*****************************************************************************/
uint32 Text8::size_in_bytes()
{
	return size_bytes;
}

/*****************************************************************************/
void Text8::print()
{
	printf("%s",text);
}

/*****************************************************************************/
void Text8::print(FILE *out)
{
	fprintf(out,"%s",text);
}

/*****************************************************************************/
bool Text8::storable_to_blob(Blob *b)
{
	uint32 id=3;
	uint32 sb=size_in_bytes();
	b->add_bytes((const uint8*)(&id),4);
	b->add_bytes((const uint8*)(&sb),4);
	b->add_bytes((const uint8*)text,sb);
	b->word_align();
	return false;
}

/*****************************************************************************/

