#include <stdio.h>
#include <stdlib.h>
#include <sys/time.h>

#include "offermon/writer.h"
#include "offermon/streams.h"
#include "offermon/tracker.h"

Writer *writer_now=NULL;
Writer *writer_old=NULL;

int NODENO;

#define TMP_SIZE 10240
unsigned char TMP[TMP_SIZE];

/**
 * Function: writers_init
 */
void writers_init()
{
	writer_old=NULL;
	int fn=1;
	bool flag=true;
	char tmp[1024];
	struct stat buf;
	while (flag) {
		sprintf(tmp,"/www/combine-erp/SITES/offermon/tracker-datafiles/%08d-%02d.dat",fn,NODENO);
		if (stat(tmp,&buf)==0) fn++; else flag=false;
	}
	printf("writer_now = %d-%d.dat\n",fn,NODENO);
	writer_now=new Writer(NODENO,fn,WRITER_APPEND);
}

/**
 * Function: writers_stop
 */
void writers_stop()
{
	delete writer_now;
	if (writer_old) delete writer_old;
}

/**
 * Function: writer_get
 */
Writer *writer_get(int fileno)
{
	if (writer_now) {
		if (writer_now->fileno==fileno) return writer_now;
	}
	if (writer_old) {
		if (writer_old->fileno==fileno) return writer_old;
		delete writer_old;
	}
	printf("*** SWITCHING OLD FILE TO %d\n",fileno);
	writer_old=new Writer(NODENO,fileno,WRITER_MODIFY);
	return writer_old;
}

/**
 * Method: Writer::new
 */
Writer::Writer(int nodeno,int fileno,int _mode)
{
	this->nodeno=nodeno;
	this->fileno=fileno;
	this->mode=_mode;
	init();
}

/**
 * Method: Writer::delete
 */
Writer::~Writer()
{
	file_close();
}

/**
 * Method: Writer::init
 */
void Writer::init()
{
	struct stat buf;
	int64 pos;
	sprintf(filename,"/www/combine-erp/SITES/offermon/tracker-datafiles/%08d-%02d.dat",fileno,nodeno);
	if (stat(filename,&buf)!=0) {
		if (mode!=WRITER_APPEND) return;
		f=fopen(filename,"w");
		fclose(f);
	}
	if (mode==WRITER_READ) {
		f=fopen(filename,"r");
	} else {
		f=fopen(filename,"r+");
	}
	if (!f) {
		fprintf(stderr,"Cannot open file %s\n",filename);
		exit(1);
	}
	fseek(f,0,SEEK_END);
	pos=ftell(f);
	if (mode==WRITER_APPEND) {
		if (pos==0) file_create(); else file_open();
	} else {
		file_open();
	}
	index_cluster_read(0);

}

/**
 * Method: Writer::opened
 */
bool Writer::opened()
{
	return f?true:false;
}

/**
 * Method: Writer::file_create
 */
void Writer::file_create()
{
	int i;
	CLUSTER_ITEMS	=DEFAULT_CLUSTER_ITEMS;
	CLUSTERS_COUNT	=DEFAULT_CLUSTERS_COUNT;
	FILE_VERSION	=DEFAULT_FILE_VERSION;
	memset(file_header,0,FILE_HEADER);
	file_header[0]='O';
	file_header[1]='F';
	file_header[2]='R';
	file_header[3]='M';

	((int32*)file_header)[1]=CLUSTER_ITEMS;
	((int32*)file_header)[2]=CLUSTERS_COUNT;
	((int32*)file_header)[3]=FILE_VERSION;

	fseek(f,0,SEEK_SET);
	fwrite(file_header,1,FILE_HEADER,f);

	for (i=0;i<CLUSTERS_COUNT;i++) {
		index_cluster_create(i);
		if (i==0) cluster[0]=CLUSTERS_COUNT*CLUSTER_ITEMS*8+FILE_HEADER;
		index_cluster_write();
	}
	maxpos=0;
}

/**
 * Method: Writer::file_close
 */
void Writer::file_close()
{
	if (f) fclose(f);
	f=NULL;
}

/**
 * Method: Writer::file_open
 */
void Writer::file_open()
{
	fseek(f,0,SEEK_SET);
	fread(file_header,1,FILE_HEADER,f);
	if (strncmp(file_header,"OFRM",4)!=0) {
		fprintf(stderr,"Writer.file_open() file signature error\n");
		fclose(f);
		f=NULL;
		return;
	}
	CLUSTER_ITEMS	=((int32*)file_header)[1];
	CLUSTERS_COUNT	=((int32*)file_header)[2];
	FILE_VERSION	=((int32*)file_header)[3];
	if (CLUSTER_ITEMS>MAX_CLUSTER_ITEMS) {
		fprintf(stderr,"Writer.file_open() CLUSTER_ITEMS is too big (%d)\n",CLUSTER_ITEMS);
		fclose(f);
		f=NULL;
		return;
	}
	if (mode!=WRITER_APPEND) {
		index_cluster_read(0);
		if (FILE_VERSION<DEFAULT_FILE_VERSION) {
			file_upgrade();
	
		} else if (FILE_VERSION>DEFAULT_FILE_VERSION) {
			fprintf(stderr,"Writer.file_open() - cannot work with file version %d (max is can work with is %d)\n",FILE_VERSION,DEFAULT_FILE_VERSION);
			fclose(f);
			f=NULL;
			return;
		}
		return;
	} else {
		if (FILE_VERSION!=DEFAULT_FILE_VERSION) {
			fprintf(stderr,"Writer.file_open() - strange things happening! Abort! Abort! (%d vs %d)\n",FILE_VERSION,DEFAULT_FILE_VERSION);
			exit(1);
		}
	}

	int i,j;

	maxpos=-1;
	cluster_pos=0;

	while (cluster_pos<CLUSTERS_COUNT && maxpos==-1) {
		index_cluster_read(cluster_pos);
		for (j=0;j<CLUSTER_ITEMS && maxpos==-1;j++) {
			if (cluster[j]==0) maxpos=j+i*CLUSTER_ITEMS;
		}
		if (maxpos==-1) cluster_pos++;
	}
}

/**
 * Method: Writer::file_upgrage
 */
void Writer::file_upgrade()
{
	char tmp1[1024];
	char tmp2[1024];
	char tmp3[1024];
	sprintf(tmp1,"/www/combine-erp/SITES/offermon/tracker-datafiles/%08d-%02d.dat",fileno,nodeno);
	sprintf(tmp2,"/www/combine-erp/SITES/offermon/tracker-datafiles/%08d-%02d.dat.old",fileno,nodeno);
	sprintf(tmp3,"/www/combine-erp/SITES/offermon/tracker-datafiles/%08d-%02d.dat",fileno,nodeno+100);
	printf("Writer::file_upgrade(nodeno=%d,fileno=%d)\n",nodeno,fileno);
	Writer *wr=new Writer(nodeno+100,fileno,WRITER_APPEND);
	int32 pos=0;
	Tracker *message;
	while ((message=read_message(pos,NULL))) {
//		printf("msg %d\n",pos);
		wr->write_message(message);
		delete message;
		pos++;
	}
	delete wr;
	file_close();
	rename(tmp1,tmp2);
	rename(tmp3,tmp1);
	init();
}

/**
 * Method: Writer::index_cluster_create
 */
void Writer::index_cluster_create(int pos)
{
	int i;
	cluster_pos=pos;
	for (i=0;i<CLUSTER_ITEMS;i++) cluster[i]=0;
}

/**
 * Method: Writer::index_cluster_read
 */
void Writer::index_cluster_read(int _cl)
{
	if (!f) return;
	cluster_pos=_cl;
	fseek(f,cluster_pos*CLUSTER_ITEMS*8+FILE_HEADER,SEEK_SET);
	fread(cluster,8,CLUSTER_ITEMS,f);
}

/**
 * Method: Writer::index_cluster_write
 */
void Writer::index_cluster_write()
{
	if (mode!=WRITER_APPEND) {
		fprintf(stderr,"Writer::index_cluster_write() nodeno=%d fileno=%d - readonly!\n",nodeno,fileno);
		return;
	}
	fseek(f,cluster_pos*CLUSTER_ITEMS*8+FILE_HEADER,SEEK_SET);
	fwrite(cluster,8,CLUSTER_ITEMS,f);
}

/**
 * Method: Writer::get_offset_of
 */
int64 Writer::get_offset_of(int i)
{
	int i1,i2;
	if (i<0 || i>=CLUSTER_ITEMS*CLUSTERS_COUNT) {
		fprintf(stderr,"Writer.get_offset_of() - illegal offset %d (file=%d,maxpos=%d)\n", i,fileno,maxpos);
		exit(1);
	}
	i1=i&127;
	i2=i>>7;
	if (i2!=cluster_pos) index_cluster_read(i2);
	return cluster[i1];
}

/**
 * Method: Writer::set_offset_of
 */
void Writer::set_offset_of(int i,int64 off)
{
	if (mode!=WRITER_APPEND) {
		fprintf(stderr,"Writer::set_offset_of() nodeno=%d fileno=%d - readonly!\n",nodeno,fileno);
		return;
	}
	int i1,i2;
	if (i<0 || i>=CLUSTER_ITEMS*CLUSTERS_COUNT) {
		fprintf(stderr,"Writer.set_offset_of() - illegal offset %d\n", i);
		exit(1);
	}
	i1=i&127;
	i2=i>>7;
	if (i2!=cluster_pos) index_cluster_read(i2);
	cluster[i1]=off;
	index_cluster_write();
}

/**
 * Method: Writer::write_message
 */
bool Writer::write_message(Tracker *m)
{
	if (mode!=WRITER_APPEND) {
		fprintf(stderr,"Writer::write_message() nodeno=%d fileno=%d - readonly!\n",nodeno,fileno);
		return false;
	}

	int64 offset;

//	printf("fileno=%d maxpos=%d MAX=%d\n",fileno,maxpos,CLUSTER_ITEMS*CLUSTERS_COUNT);
	if (maxpos>=CLUSTER_ITEMS*CLUSTERS_COUNT-1) return false;
	if (maxpos<0) return false;

	offset=get_offset_of(maxpos);
	fseek(f,offset,SEEK_SET);

	WriteStream *ws=new WriteStream(TMP,TMP_SIZE);

	ws->write_int64(m->offer_id);					// 0
	ws->write_int64(m->user_id);					// 8
	ws->write_int64(m->smartlink_offer_id);			// 16
	ws->write_int64(m->user_agent_id);				// 24
	ws->write_int64(m->js_user_agent_id);			// 32
	ws->write_int32(m->checks);						// 40
	ws->write_int32(m->js_width);					// 44
	ws->write_int32(m->js_height);					// 48
	ws->write_int32(m->payout_db*10000);			// 52
	ws->write_int32(m->payout_db_usd*10000);		// 56
	ws->write_int32(m->payout_db_aff*10000);		// 60
	ws->write_int32(m->currency_db_id);				// 64
	ws->write_int32(m->payout_http*10000);			// 68
	ws->write_int64(m->payout_user_agent_id);		// 72
	ws->write_int64(m->click_date);					// 80
	ws->write_int64(m->payout_date);				// 88
	ws->write_int32(m->gif_rnd);					// 96
	ws->write_int32(m->region_id);					// 100
	ws->write_int32(m->device_id);					// 104
	ws->write_ip(m->ip);							// 108
	ws->write_ip(m->payout_ip);						// 108+12
													// 108+12+12

	ws->write_charptr(m->foreign_clickid);
	if (m->aff_source) { ws->write_int32(AFF_SOURCE);ws->write_charptr(m->aff_source); }
	if (m->aff_sub   ) { ws->write_int32(AFF_SUB   );ws->write_charptr(m->aff_sub   ); }
	if (m->aff_sub2  ) { ws->write_int32(AFF_SUB2  );ws->write_charptr(m->aff_sub2  ); }
	if (m->aff_sub3  ) { ws->write_int32(AFF_SUB3  );ws->write_charptr(m->aff_sub3  ); }
	if (m->aff_sub4  ) { ws->write_int32(AFF_SUB4  );ws->write_charptr(m->aff_sub4  ); }
	if (m->aff_sub5  ) { ws->write_int32(AFF_SUB5  );ws->write_charptr(m->aff_sub5  ); }
	if (m->aff_pb    ) { ws->write_int32(AFF_PB    );ws->write_charptr(m->aff_pb    ); }
	if (m->referer   ) { ws->write_int32(REFERER   );ws->write_charptr(m->referer   ); }
	ws->write_int32(0);

/*	FILE *tmpfile;
	tmpfile=fopen("test.dat","w");
	fwrite(ws->src_ptr,ws->len(),1,tmpfile);
	fclose(tmpfile);
	printf("source=%s\n",m->aff_source);
	exit(1);*/

	fwrite(ws->src_ptr,ws->len(),1,f);
	delete ws;
	m->clickid_entryno	=maxpos;
	m->clickid_fileno	=fileno;
	m->clickid_nodeno	=nodeno;
	// Запись сохранили, вычисляем результирующий offset и сохраняем в индекс
	maxpos++;
	offset=ftell(f);
	set_offset_of(maxpos,offset);

	return true;
}

/**
 * Method: Writer::read_message
 */
Tracker *Writer::read_message(int pos,Tracker *m)
{
	if (!f) {
		fprintf(stderr,"Writer::read_message() f==NULL\n");
		return NULL;
	}
	if (pos+1>=CLUSTER_ITEMS*CLUSTERS_COUNT) {
//		fprintf(stderr,"Writer::read_message() pos+1>=CLUSTER_ITEMS*CLUSTERS_COUNT (%d > %d*%d)!\n",pos+1,CLUSTER_ITEMS,CLUSTERS_COUNT);
		return NULL;
	}
	int64 offset=get_offset_of(pos);
	int64 offset2=get_offset_of(pos+1);
	if (offset==0 || offset2==0) {
		fprintf(stderr,"Writer::read_message() offset==0 || offset2==0 (nodeno=%d, fileno=%d, pos=%d offset=%ld, offset2=%ld)\n",nodeno,fileno,pos,offset,offset2);
		return NULL;
	}
	if (offset==offset2) {
		fprintf(stderr,"Writer::read_message() offset==offset2 (%ld)\n",offset);
		return NULL;
	}

	if (m==NULL) {
		m=new Tracker();
		m->init();
	}
	unsigned char *tmp=(unsigned char*)malloc(offset2-offset);
	fseek(f,offset,SEEK_SET);
	fread(tmp,offset2-offset,1,f);

	switch (FILE_VERSION) {
		case 0:
			return read_message_v0(pos,m,offset,offset2,tmp);
		case 1:
			return read_message_v1(pos,m,offset,offset2,tmp);
		default:
			fprintf(stderr,"Writer::read_message() Cannot read message of version %d!\n",FILE_VERSION);
			return NULL;
	}
}

/**
 * Method: Writer::read_message_v0
 */
Tracker *Writer::read_message_v0(int pos,Tracker *m,int64 offset,int64 offset2,unsigned char *tmp)
{
	int aff_fld;
//	printf("Writer::read_message(%d) - reading %d bytes at pos %d\n",pos,offset2-offset,offset);
	ReadStream *st=new ReadStream(tmp,offset2-offset);

	m->offer_id				=st->read_int64();	// 0
	m->user_id				=st->read_int64();	// 8
	m->smartlink_offer_id	=st->read_int64();	// 16
	m->user_agent_id		=st->read_int64();	// 24
	m->js_user_agent_id		=st->read_int64();	// 32
	m->checks				=st->read_int32();	// 40
	m->js_width				=st->read_int32();	// 44
	m->js_height			=st->read_int32();	// 48
	m->payout_http			=st->read_int32()/10000.0;
	m->payout_user_agent_id	=st->read_int64();
	m->click_date			=st->read_int64();
	m->payout_date			=st->read_int64();
	m->gif_rnd				=st->read_int32();
	m->region_id			=st->read_int32();
	m->device_id			=st->read_int32();
							 st->read_ip(m->ip);
							 st->read_ip(m->payout_ip);

	m->foreign_clickid		=st->read_charptr();
	
	m->clickid_entryno		=pos;
	m->clickid_fileno		=fileno;
	m->clickid_nodeno		=nodeno;

	aff_fld=st->read_int32();
	while (aff_fld) {
		char *tmp=st->read_charptr();
		switch(aff_fld) {
			case AFF_SOURCE:m->aff_source=tmp;break;
			case AFF_SUB   :m->aff_sub   =tmp;break;
			case AFF_SUB2  :m->aff_sub2  =tmp;break;
			case AFF_SUB3  :m->aff_sub3  =tmp;break;
			case AFF_SUB4  :m->aff_sub4  =tmp;break;
			case AFF_SUB5  :m->aff_sub5  =tmp;break;
			case AFF_PB    :m->aff_pb    =tmp;break;
		}
		aff_fld=st->read_int32();
	}
	delete st;
	m->free_on_delete=tmp;

	m->calc_db_payouts(NULL);
	return m;
}

/**
 * Method: Writer::read_message_v1
 */
Tracker *Writer::read_message_v1(int pos,Tracker *m,int64 offset,int64 offset2,unsigned char *tmp)
{
	int aff_fld;
//	printf("Writer::read_message(%d) - reading %d bytes at pos %d\n",pos,offset2-offset,offset);
	ReadStream *st=new ReadStream(tmp,offset2-offset);

	m->offer_id				=st->read_int64();
	m->user_id				=st->read_int64();
	m->smartlink_offer_id	=st->read_int64();
	m->user_agent_id		=st->read_int64();
	m->js_user_agent_id		=st->read_int64();
	m->checks				=st->read_int32();
	m->js_width				=st->read_int32();
	m->js_height			=st->read_int32();
	m->payout_db			=st->read_int32()/10000.0;
	m->payout_db_usd		=st->read_int32()/10000.0;
	m->payout_db_aff		=st->read_int32()/10000.0;
	m->currency_db_id		=st->read_int32();
	m->payout_http			=st->read_int32()/10000.0;
	m->payout_user_agent_id	=st->read_int64();
	m->click_date			=st->read_int64();
	m->payout_date			=st->read_int64();
	m->gif_rnd				=st->read_int32();
	m->region_id			=st->read_int32();
	m->device_id			=st->read_int32();
							 st->read_ip(m->ip);
							 st->read_ip(m->payout_ip);

	m->foreign_clickid		=st->read_charptr();
	
	m->clickid_entryno		=pos;
	m->clickid_fileno		=fileno;
	m->clickid_nodeno		=nodeno;

	aff_fld=st->read_int32();
	while (aff_fld) {
		char *tmp=st->read_charptr();
		switch(aff_fld) {
			case AFF_SOURCE:m->aff_source=tmp;break;
			case AFF_SUB   :m->aff_sub   =tmp;break;
			case AFF_SUB2  :m->aff_sub2  =tmp;break;
			case AFF_SUB3  :m->aff_sub3  =tmp;break;
			case AFF_SUB4  :m->aff_sub4  =tmp;break;
			case AFF_SUB5  :m->aff_sub5  =tmp;break;
			case AFF_PB    :m->aff_pb    =tmp;break;
			case REFERER   :m->referer   =tmp;break;
		}
		aff_fld=st->read_int32();
	}
	delete st;
	m->free_on_delete=tmp;
	return m;
}


/**
 * Method: Writer::set_message_flag
 */
void Writer::set_message_flag(Tracker *m)
{
	if (FILE_VERSION!=DEFAULT_FILE_VERSION) {
		fprintf(stderr,"Writer::set_message_flag() Cannot modify message, file versions differ!\n");
		return;
	}
	int64 offset=get_offset_of(m->clickid_entryno);
	int64 offset2=get_offset_of(m->clickid_entryno+1);
	if (offset==0 || offset2==0) return;

	unsigned char *tmp=(unsigned char*)malloc(offset2-offset);
	bool need_write=false;

	fseek(f,offset,SEEK_SET);
	fread(tmp,offset2-offset,1,f);

	WriteStream *ws=new WriteStream(tmp,offset2-offset);
	switch (m->mode) {
		case MODE_GIF:
			ws->set(96);// TODO GIF POSITION
			if (ws->read_int32()==m->gif_rnd) {
				ws->set(40);// TODO FLAG POSITION
				int checks=ws->read_int32();
				checks|=FLAG_GIF;
				ws->set(40);// TODO FLAG POSITION
				ws->write_int32(checks);
				need_write=true;
			}
			break;
		case MODE_AJAX:
			ws->set(32);// TODO JS_USER_AGENT_ID POSITION
			ws->write_int64(m->js_user_agent_id);
			ws->set(44);// TODO JS_WIDTH POSITION
			ws->write_int32(m->js_width);
			ws->set(48);// TODO JS_HEIGHT POSITION
			ws->write_int32(m->js_height);

			ws->set(40);// TODO FLAG POSITION
			int checks=ws->read_int32();
			checks|=FLAG_AJAX;
			ws->set(40);// TODO FLAG POSITION
			ws->write_int32(checks);
			need_write=true;
			break;
	}

	if (need_write) {
		fseek(f,offset,SEEK_SET);
		fwrite(tmp,offset2-offset,1,f);
	}

	delete ws;
	free(tmp);
}

/**
 * Method: Writer::set_message_payout
 */
bool Writer::set_message_payout(Tracker *m)
{
	if (FILE_VERSION!=DEFAULT_FILE_VERSION) {
		fprintf(stderr,"Writer::set_message_flag() Cannot modify message, file versions differ!\n");
		return false;
	}
	int aff_fld;
	int64 offset=get_offset_of(m->clickid_entryno);
	int64 offset2=get_offset_of(m->clickid_entryno+1);
	if (offset==0 || offset2==0) return false;

	unsigned char *tmp=(unsigned char*)malloc(offset2-offset);

	fseek(f,offset,SEEK_SET);
	fread(tmp,offset2-offset,1,f);

	WriteStream *ws=new WriteStream(tmp,offset2-offset);
	
	struct timeval tv;
	struct timezone tz;
	gettimeofday(&tv,&tz);
	m->payout_date=tv.tv_sec;

	ws->set(68);// TODO PAYOUT HTTP POSITION
	ws->write_int32(10000.0*m->payout_http);		// 68
	ws->write_int64(m->payout_user_agent_id);		// 72

	ws->set(88);// TODO PAYOUT DATE POSITION
	ws->write_int64(m->payout_date);				// 88

	ws->set(108+12);// TODO PAYOUT IP POSITION
	ws->write_ip(m->payout_ip);						// 108+12
	

	fseek(f,offset,SEEK_SET);
	fwrite(tmp,offset2-offset,1,f);

	delete ws;
	free(tmp);

	return true;
}

/**
 * Method: Writer::get_maxpos
 */
int Writer::get_maxpos()
{
	return maxpos;
}


